/* WTF Module
 *   Stuff
 *
 * by Steve Dudenhoeffer
 *
 * This file is part of WTF Module
 *
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at
 *  your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *  In addition, as a special exception, the author gives permission to
 *  link the code of this program with the Half-Life Game Engine ("HL
 *  Engine") and Modified Game Libraries ("MODs") developed by Valve,
 *  L.L.C ("Valve"). You must obey the GNU General Public License in all
 *  respects for all of the code used other than the HL Engine and MODs
 *  from Valve. If you modify this file, you may extend this exception
 *  to your version of the file, but you are not obligated to do so. If
 *  you do not wish to do so, delete this exception statement from your
 *  version.
 */

#include "weaponmod.h"

/* Taken mostly from HLSDK */

void CSnark::Die()
{
	pev->takedamage = DAMAGE_NO;
	EMIT_SOUND_DYN(ENT(pev), CHAN_ITEM, m_BlastSound.c_str("squeek/sqk_blast1.wav"), 1, 0.5, 0, PITCH_NORM);	
	UTIL_BloodDrips( pev->origin, Vector(0,0,0), BLOOD_COLOR_YELLOW, 80 );

	// Make blood decals on walls near it 
	TraceResult tr;
	ClosestWall(&tr, 8, 40);
	switch(RANDOM_LONG(0,5))
	{
	case 0:
		UTIL_DecalTrace(&tr,"{yblood1");
		break;
	case 1:
		UTIL_DecalTrace(&tr,"{yblood2");
		break;
	case 2:
		UTIL_DecalTrace(&tr,"{yblood3");
		break;
	case 3:
		UTIL_DecalTrace(&tr,"{yblood4");
		break;
	case 4:
		UTIL_DecalTrace(&tr,"{yblood5");
		break;
	case 5:
		UTIL_DecalTrace(&tr,"{yblood6");
		break;
	};
	pev->owner=NULL;
	if (m_hOwner != NULL)
		pev->owner=m_hOwner;
	String m_Temp;
	m_Temp.assign(m_Classname.c_str("snark"));
	game->RadialDamage(edict(), &m_Temp, pev->dmg, pev->dmg);
	Remove();
}
void CSnark::Setup(edict_t *owner)
{
	m_DamageBite=10.0;
	m_Damage=25.0; 
	m_Life=10.0;  // Lifespan
	pev->owner=owner;
	m_Health=10.0; // HP
	m_hOwner=pev->owner;
	m_Classname.assign("snark");
	m_Speed=500.0;
	float origin[3];
	if (!pev->owner)
		origin[0] = origin[1] = origin[2] = 0.0;
	else
	{
		origin[0]=pev->owner->v.origin[0] + pev->owner->v.view_ofs[0];
		origin[1]=pev->owner->v.origin[1] + pev->owner->v.view_ofs[1];
		origin[2]=pev->owner->v.origin[2] + pev->owner->v.view_ofs[2];
	}
	float vel[3];
	vel[0]=pev->owner->v.v_angle[0] + pev->owner->v.punchangle[0];
	vel[1]=pev->owner->v.v_angle[1] + pev->owner->v.punchangle[1];
	vel[2]=pev->owner->v.v_angle[2] + pev->owner->v.punchangle[2];
	MAKE_VECTORS(vel);
	pev->angles[0] = pev->owner->v.v_angle[0];
	pev->angles[1] = pev->owner->v.v_angle[1];
	pev->angles[2] = pev->owner->v.v_angle[2];
	pev->angles[0] -= 30;
	pev->angles[0] = -(pev->angles[0] + 30);
	pev->velocity[0] = gpGlobals->v_forward[0] * (m_Speed * 1.66);
	pev->velocity[1] = gpGlobals->v_forward[1] * (m_Speed * 1.66);
	pev->velocity[2] = gpGlobals->v_forward[2] * (m_Speed * 1.66);
	m_LaunchSequence=3;
	SET_ORIGIN(edict(),origin);
	if (!strncmp(MF_GetModname(),"ns",2))
		pev->team=3;
		
	m_Mins[0]=-4.0;
	m_Mins[1]=-4.0;
	m_Mins[2]=0.0;
	m_Maxs[0]=4.0;
	m_Maxs[1]=4.0;
	m_Maxs[2]=8.0;

}
void CSnark::Spawn(void)
{
	// motor
	pev->movetype=MOVETYPE_BOUNCE;
	pev->solid=SOLID_BBOX;
	SET_MODEL(ENT(pev), m_SnarkModel.c_str("models/w_squeak.mdl"));
	SET_ORIGIN(edict(), pev->origin );
	SET_SIZE(edict(), Vector(m_Mins[0], m_Mins[1], m_Mins[2]), Vector(m_Maxs[0], m_Maxs[1], m_Maxs[2]));

	CSetTouch( &CSnark::BounceTouch );
	CSetThink( &CSnark::HuntThink );
	pev->nextthink = gpGlobals->time + 0.1;
	m_flNextHunt = gpGlobals->time + 1E6;

	pev->flags |= FL_MONSTER;
	pev->takedamage		= DAMAGE_YES;
	pev->health			= m_Health + HURTABLE_HEALTH_BUFFER;
	pev->gravity		= 0.5;
	pev->friction		= 0.5;

	pev->dmg = m_Damage;

	m_flDie = gpGlobals->time + m_Life;

	m_flFieldOfView = 0; // 180 degrees

	if ( pev->owner )
		m_hOwner = pev->owner;

	m_flNextBounceSoundTime = gpGlobals->time;// reset each time a snark is spawned.
	m_hEnemy=NULL;
	pev->sequence = m_LaunchSequence;
	pev->framerate = 1.0;
}
void CSnark::HuntThink()
{

	if (!IsInWorld()) {
		CSetTouch( NULL );
		Remove();
		return;
	}
	
	FrameAdvance();
	pev->nextthink = gpGlobals->time + 0.1;

	// explode when ready
	if (gpGlobals->time >= m_flDie)
	{
		g_vecAttackDir = pev->velocity.Normalize( );
		pev->health = -1;
		Die();
		return;
	}

	// float
	if (pev->waterlevel != 0)
	{
		if (pev->movetype == MOVETYPE_BOUNCE)
		{
			pev->movetype = MOVETYPE_FLY;
		}
		pev->velocity = pev->velocity * 0.9;
		pev->velocity.z += 8.0;
	}
	else if (pev->movetype == MOVETYPE_FLY)
	{
		pev->movetype = MOVETYPE_BOUNCE;
	}

	// return if not time to hunt
	if (m_flNextHunt > gpGlobals->time)
		return;

	m_flNextHunt = gpGlobals->time + 2.0;
	
	edict_t *pOther = NULL;
	Vector vecDir;
	TraceResult tr;

	Vector vecFlat = pev->velocity;
	vecFlat.z = 0;
	vecFlat = vecFlat.Normalize( );

	UTIL_MakeVectors( pev->angles );

	if (m_hEnemy == NULL || m_hEnemy->v.deadflag == DEAD_NO || m_hEnemy->v.health <= 0)
	{
		// find target, bounce a bit towards it.
		Look( 512 );
		m_hEnemy = FindBestEnemy( );
	}

	// squeek if it's about time blow up
	if ((m_flDie - gpGlobals->time <= 0.5) && (m_flDie - gpGlobals->time >= 0.3))
		EMIT_SOUND_DYN(ENT(pev), CHAN_VOICE, m_DieSound.c_str("squeek/sqk_die1.wav"), 1, ATTN_NORM, 0, 100 + RANDOM_LONG(0,0x3F));

	// higher pitch as squeeker gets closer to detonation time
	float flpitch = 155.0 - 60.0 * ((m_flDie - gpGlobals->time) / m_Life);
	if (flpitch < 80)
		flpitch = 80;

	if (m_hEnemy != NULL)
	{
		if (FVisible( m_hEnemy ))
		{
			vecDir = (m_hEnemy->v.origin + m_hEnemy->v.view_ofs) - pev->origin;
			m_vecTarget = vecDir.Normalize( );
		}

		float flVel = pev->velocity.Length();
		float flAdj = 50.0 / (flVel + 10.0);

		if (flAdj > 1.2)
			flAdj = 1.2;
		
		pev->velocity = pev->velocity * flAdj + m_vecTarget * m_Speed;
	}

	if (pev->flags & FL_ONGROUND)
	{
		pev->avelocity = Vector( 0, 0, 0 );
	}
	else
	{
		if (pev->avelocity == Vector( 0, 0, 0))
		{
			pev->avelocity.x = RANDOM_FLOAT( -100, 100 );
			pev->avelocity.z = RANDOM_FLOAT( -100, 100 );
		}
	}

	if ((pev->origin - m_posPrev).Length() < 1.0)
	{
		pev->velocity.x = RANDOM_FLOAT( -100, 100 );
		pev->velocity.y = RANDOM_FLOAT( -100, 100 );
	}
	m_posPrev = pev->origin;

	pev->angles = UTIL_VecToAngles( pev->velocity );
	pev->angles.z = 0;
	pev->angles.x = 0;
}
void CSnark::Look(int iDistance)
{
	int	iSighted = 0;

	int cur=0;
	while (cur<99) {
		m_Potentials[cur]=NULL;
		cur++;
	}
	cur=0;
	m_pLink = NULL;

	edict_t	*pSightEnt = NULL;// the current visible entity that we're dealing with

	edict_t *pList[100];

	Vector delta = Vector( iDistance, iDistance, iDistance );

	// Find only monsters/clients in box, NOT limited to PVS
	int count = UTIL_EntitiesInBox( pList, 100, pev->origin - delta, pev->origin + delta, FL_CLIENT|FL_MONSTER );
	for ( int i = 0; i < count; i++ )
	{
		pSightEnt = pList[i];
		if ( pSightEnt != edict() && pSightEnt->v.health > 0 )
		{
			if ( pSightEnt->v.modelindex != pev->modelindex && FInViewCone( pSightEnt ) && !FBitSet( pSightEnt->v.flags, FL_NOTARGET ) && FVisible( pSightEnt ) )
			{
				m_Potentials[cur]=pSightEnt;
				cur++;
			}
		}
	}
}
void CSnark::BounceTouch(edict_t *pOther)
{
	float	flpitch;

	TraceResult tr = UTIL_GetGlobalTrace( );

	// don't hit the guy that launched this grenade
	if ( pev->owner && pOther == pev->owner )
		return;

	// at least until we've bounced once
	pev->owner = NULL;
	pev->angles=UTIL_VecToAngles(pev->velocity);
	pev->angles.x = 0;
	pev->angles.z = 0;

	// avoid bouncing too much
	if (m_flNextHit > gpGlobals->time)
		return;

	// higher pitch as squeeker gets closer to detonation time
	flpitch = 155.0 - 60.0 * ((m_flDie - gpGlobals->time) / m_Life);

	if ( pOther->v.takedamage && m_flNextAttack < gpGlobals->time )
	{
		// attack!

		// make sure it's me who has touched them
		if (tr.pHit == pOther)
		{
			// and it's not another squeakgrenade
			if (tr.pHit->v.modelindex != pev->modelindex)
			{
				String m_Temp;
				m_Temp.assign(m_Classname.c_str("snark"));
	
				if (m_hOwner)
					game->TakeDamage(pOther,m_hOwner, &m_Temp,m_DamageBite); 
				else
					game->TakeDamage(pOther,pOther,&m_Temp,m_DamageBite);

				pev->dmg += m_Damage; // add more explosion damage
				// m_flDie += 2.0; // add more life

				// make bite sound
				EMIT_SOUND_DYN(ENT(pev), CHAN_WEAPON, m_BiteSound.c_str("squeek/sqk_deploy1.wav"), 1.0, ATTN_NORM, 0, (int)flpitch);
				m_flNextAttack = gpGlobals->time + 0.5;
			}
		}
		else
		{
			// ALERT( at_console, "been hit\n");
		}
	}

	m_flNextHit = gpGlobals->time + 0.1;
	m_flNextHunt = gpGlobals->time;

	// in multiplayer, we limit how often snarks can make their bounce sounds to prevent overflows.
	if ( gpGlobals->time < m_flNextBounceSoundTime )
	{
		// too soon!
		return;
	}

	if (!(pev->flags & FL_ONGROUND))
	{
		// play bounce sound
		float flRndSound = RANDOM_FLOAT ( 0 , 1 );

		if ( flRndSound <= 0.33 )
			EMIT_SOUND_DYN(ENT(pev), CHAN_VOICE, m_HuntSound1.c_str("squeek/sqk_hunt1.wav"), 1, ATTN_NORM, 0, (int)flpitch);		
		else if (flRndSound <= 0.66)
			EMIT_SOUND_DYN(ENT(pev), CHAN_VOICE, m_HuntSound2.c_str("squeek/sqk_hunt2.wav"), 1, ATTN_NORM, 0, (int)flpitch);
		else 
			EMIT_SOUND_DYN(ENT(pev), CHAN_VOICE, m_HuntSound3.c_str("squeek/sqk_hunt3.wav"), 1, ATTN_NORM, 0, (int)flpitch);
	}

	m_flNextBounceSoundTime = gpGlobals->time + 0.5;// half second.
}

BOOL CSnark::FInViewCone ( edict_t *pEntity )
{
	Vector2D	vec2LOS;
	float	flDot;

	UTIL_MakeVectors ( pev->angles );
	
	vec2LOS = ( pEntity->v.origin - pev->origin ).Make2D();
	vec2LOS = vec2LOS.Normalize();

	flDot = DotProduct (vec2LOS , gpGlobals->v_forward.Make2D() );

	if ( flDot > m_flFieldOfView )
	{
		return TRUE;
	}
	else
	{
		return FALSE;
	}
}
BOOL CSnark::FVisible(edict_t *pEntity)
{
	TraceResult tr;
	Vector		vecLookerOrigin;
	Vector		vecTargetOrigin;
	
	if (FBitSet( pEntity->v.flags, FL_NOTARGET ))
		return FALSE;

	// don't look through water
	if ((pev->waterlevel != 3 && pEntity->v.waterlevel == 3) 
		|| (pev->waterlevel == 3 && pEntity->v.waterlevel == 0))
		return FALSE;

	vecLookerOrigin = pev->origin + pev->view_ofs;//look through the caller's 'eyes'
	vecTargetOrigin = pEntity->v.origin + pEntity->v.view_ofs;

	UTIL_TraceLine(vecLookerOrigin, vecTargetOrigin, ignore_monsters, ignore_glass, ENT(pev)/*pentIgnore*/, &tr);
	
	if (tr.flFraction != 1.0)
	{
		return FALSE;// Line of sight is not established
	}
	else
	{
		return TRUE;// line of sight is valid.
	}
}
edict_t *CSnark::FindBestEnemy(void)
{
	edict_t	*pReturn;
	edict_t	*pNextEnt;
	float			iNearest;
	float			iDist;

	iNearest = 8192;// so first visible entity will become the closest.
	pNextEnt = m_Potentials[0];
	pReturn = NULL;
	int cur=0;
	while ( pNextEnt != NULL )
	{
		if ( pNextEnt->v.deadflag == DEAD_NO && pNextEnt->v.health > 0 )
		{
			iDist = ( pNextEnt->v.origin - pev->origin ).Length();
			
			if ( iDist <= iNearest )
			{
				iNearest = iDist;
				pReturn = pNextEnt;
			}
		}

		pNextEnt = m_Potentials[cur];
		cur++;
	}

	return pReturn;
}
